/*****************************************************************************
*
* Copyright 2019 NXP
* All Rights Reserved
*
*****************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
* THE POSSIBILITY OF SUCH DAMAGE.
*
****************************************************************************/


/****************************************************************************
* Includes
****************************************************************************/

#ifndef APEX2_EMULATE
  #include "apex.h"
#endif

#include <apexcv_base.h>


/****************************************************************************
* Main function
****************************************************************************/
void apexcv_basic()
{
  int lSrcWidth = 1024;
  int lSrcHeight = 512;

  // Allocate the input and output buffers
  vsdk::SUMat lInput0(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
  vsdk::SUMat lInput1(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
  vsdk::SUMat lOutput0(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);

  // Control output
  printf("Data on: \n");
  printf("   Input 1           (%dx%d bytes) at %p\n", lSrcWidth, lSrcHeight, lInput0.u->handle);
  printf("   Input 2           (%dx%d bytes) at %p\n", lSrcWidth, lSrcHeight, lInput1.u->handle);
  printf("   Output            (%dx%d words) at %p\n", lSrcWidth, lSrcHeight, lOutput0.u->handle);

  int lRetVal = 0;

  // Create a new add process for the Apex core.
  apexcv::Add myAdd;

  // Initialize the process, this is the mandatory first call to apexcv object
  lRetVal |= myAdd.Initialize(lInput0, lInput1, lOutput0);
  if (lRetVal)
  {
    printf("Error on Initialize: %d \n", lRetVal);
  }
  printf("Initialize Done \n");

  // Compute the results, the output buffer is updated after Process executes
  lRetVal |= myAdd.Process();
  if (lRetVal)
  {
    printf("Error on Process: %d \n", lRetVal);
  }
  printf("Process Done \n");


  printf("Success....END [pass]\n");
}


void apexcv_simple()
{
	const int lSrcWidth = 1024;
	const int lSrcHeight = 512;
	const int lHistWidth = 256;
	const int lHistHeight = 1;

	// Allocate the input and output buffers
	vsdk::SUMat lInput0(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
	vsdk::SUMat lOutput0(lHistHeight, lHistWidth, VSDK_CV_32SC1);

	// Control output
	printf("Data on: \n");
	printf("   Input             (%dx%d bytes) at %p\n", lSrcWidth, lSrcHeight, lInput0.u->handle);
	printf("   Output            (%dx%d words) at %p\n", lSrcWidth, lSrcHeight, lOutput0.u->handle);

	int lRetVal = 0;

	// Create a new process for the Apex core.
	apexcv::Histogram myAdd;

	// Initialize the process, mandatory first call to the object
	lRetVal |= myAdd.Initialize(lInput0, lOutput0);
	if (lRetVal)
	{
	  printf("Error on Initialize: %d \n", lRetVal);
	}
	printf("Initialize Done \n");


	// Compute the results, call after Initialize for getting result
	lRetVal |= myAdd.Process();
	if (lRetVal)
	{
	  printf("Error on Process: %d \n", lRetVal);
	}
	printf("Single Process Done \n");


	// For processing in a loop and reduce overhead
	// call reconnectIO and then Process
	vsdk::SUMat lInputA(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
	vsdk::SUMat lOutputA(lHistHeight, lHistWidth, VSDK_CV_32SC1);
	vsdk::SUMat lInputB(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
	vsdk::SUMat lOutputB(lHistHeight, lHistWidth, VSDK_CV_32SC1);
	const int cLoopCount = 10;

	for (int i = 1; i <= cLoopCount; i++)
	{
	  printf("Run %d/%d in loop ", i, cLoopCount);

	  vsdk::SUMat lOutput;
	  vsdk::SUMat lInput;

	  // alternate buffers on each iteration
	  if (i & 0x01)
	  {
	    lOutput = lOutputA;
	    lInput = lInputA;
	    printf("using buffers A \n");
	  }
	  else
	  {
	    lOutput = lOutputB;
	    lInput = lInputB;
	    printf("using buffers B \n");
	  }


	  // Change the buffer to use for the processing
	  lRetVal |= myAdd.ReconnectIO(lInput, lOutput);
	  if (lRetVal)
	  {
	    printf("Error on ReconnectIO: %d \n", lRetVal);
	  }
	  printf("ReconnectIO Done \n");

	  // Compute the results
	  lRetVal |= myAdd.Process();
	  if (lRetVal)
	  {
	    printf("Error on Process: %d \n", lRetVal);
	  }
	  printf("Process Done \n");
	}

	printf("Success....END [pass]\n");
}

void APU_CALL()
{
#if __GNUG__
  setvbuf(stdout, NULL, _IONBF, 0);
  setvbuf(stderr, NULL, _IONBF, 0);
#endif

  printf("APEXCV_BASIC example\n\n");
  apexcv_basic();
  printf("\nAPEXCV_SIMPLE example\n\n");
  apexcv_simple();
}

